<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CarModelController extends Controller
{
    public function index()
    {
        $data = DB::table('car_models as m')
            ->leftJoin('car_brands as b', 'b.id', '=', 'm.brand_id')
            ->leftJoin('cars as c', 'm.id', '=', 'c.model_id')
            ->where('b.is_active', true)
            ->whereNull('m.deleted_at')
            ->select(
                'm.id',
                'm.name',
                'm.slug',
                'm.is_active',
                'b.name AS brandName',
                'm.brand_id AS brandId',
                DB::raw('COUNT(c.id) as carsCount')
            )
            ->groupBy('m.id', 'm.name', 'm.slug', 'm.is_active', 'm.brand_id', 'b.name')
            ->orderBy('m.id', 'desc')
            ->get();

        return $this->sendResponse(['data' => $data], 'Brands retrieved', 200);
    }

    public function byBrand($brandId)
    {
        $data = DB::table('car_models')
            ->where('brand_id', $brandId)
            ->where('is_active', true)
            ->whereNull('deleted_at')
            ->get();

        return $this->sendResponse(['data' => $data], 'Models retrieved', 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'brand_id' => 'required|exists:car_brands,id',
            'models' => 'required|array|min:1',
            'models.*' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        $slugs = collect($request->models)
            ->map(fn ($m) => Str::slug($m))
            ->unique()
            ->values();

        $existingSlugs = DB::table('car_models')
            ->where('brand_id', $request->brand_id)
            ->whereIn('slug', $slugs)
            ->pluck('slug')
            ->toArray();

        $models = [];

        foreach ($request->models as $model) {
            $slug = Str::slug($model);
            $existModel = DB::table('car_models')
                      ->where('slug', Str::slug($model))
                      ->first();
            if ($existModel && $existModel->id) {
                DB::table('car_models')
                ->where('id', $existModel->id)
                ->update(['deleted_at' => null]);
            } else {
                if (!in_array($slug, $existingSlugs)) {
                    $models[] = [
                        'brand_id' => $request->brand_id,
                        'name' => $model,
                        'slug' => $slug,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
            }
        }

        if (empty($models)) {
            return $this->sendError(
                'All models already exist for this brand',
                'Duplicate Models',
                422
            );
        }

        DB::table('car_models')->insert($models);

        return $this->sendResponse([
            'data' => count($models),
            'skipped' => count($request->models) - count($models),
        ], 'Models created successfully', 201);
    }


    public function editModel(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'model_id' => 'required|exists:car_models,id',
            'brand_id' => 'required|exists:car_brands,id',
            'name' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        $model = DB::table('car_models')
            ->where('id', $request->model_id)
            ->update([
                'brand_id' => $request->brand_id,
                'name' => $request->name,
                'slug' => Str::slug($request->name),
                'updated_at' => now(),
            ]);

        if (!$model) {
            return $this->sendError(
                'models not exist',
                'models not exist',
                422
            );
        }

        return $this->sendResponse([
            'data' => $model
        ], 'Models edited successfully', 201);
    }


    public function disabledOrDeleteModel(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {

            $model = DB::table('car_models')->where('id', $request->id)->first();

            if ($request->delete) {
                $modelId = DB::table('car_models')
                        ->where('id', $request->id)
                       ->update(['deleted_at' => now()]);
            } else {

                $modelId = DB::table('car_brands')
                ->where('id', $request->id)
                ->update([
                    'is_active' => $model->is_active === 0 ? 1 : 0,
                ]);
            }

            return $this->sendResponse(['data' => $modelId], 'Model created', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }

}
