<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CarBrandController extends Controller
{
    public function index()
    {
        $brands = DB::table('car_brands as b')
            ->leftJoin('cars as c', 'b.id', '=', 'c.brand_id')
            ->whereNull('b.deleted_at')
            ->select(
                'b.id',
                'b.name',
                'b.slug',
                'b.logo',
                'b.is_active',
                DB::raw('COUNT(c.id) as carsCount'),
                DB::raw('(
    SELECT ci.image_path
    FROM car_images ci
    WHERE ci.car_id = (
        SELECT ca.id
        FROM cars ca
        WHERE ca.brand_id = b.id
ORDER BY ca.id DESC
        LIMIT 1
    )
    ORDER BY ci.id ASC
    LIMIT 1
) as carImage')
            )
            ->groupBy(
                'b.id',
                'b.name',
                'b.slug',
                'b.logo',
                'b.is_active'
            )
            ->orderBy('carsCount', 'desc')
            ->get();

        $models = DB::table('car_models as m')
            ->where('m.is_active', true)
            ->whereNull('m.deleted_at')
            ->select(
                'm.id',
                'm.brand_id',
                'm.name',
                'm.slug'
            )
            ->orderBy('m.name')
            ->get()
            ->groupBy('brand_id');

        $data = $brands->map(function ($brand) use ($models) {
            $brand->models = $models[$brand->id] ?? [];

            return $brand;
        });


        return $this->sendResponse(['data' => $data], 'Brands retrieved', 200);
    }

    public function show($slug)
    {
        $brand = DB::table('car_brands')
            ->where('slug', $slug)
            ->whereNull('deleted_at')
            ->first();

        if (!$brand) {
            return $this->sendError('Brand not found', 'Not Found', 404);
        }

        $models = DB::table('car_models')
            ->where('brand_id', $brand->id)
            ->where('is_active', true)
            ->whereNull('deleted_at')
            ->get();

        $success['data'] = [
            'brand' => $brand,
            'models' => $models
        ];

        return $this->sendResponse($success, 'Brand retrieved', 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|unique:car_brands,name',
            'models' => 'nullable|array',
            'models.*' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        DB::beginTransaction();

        try {
            $brandId = DB::table('car_brands')->insertGetId([
                'name' => $request->name,
                'slug' => Str::slug($request->name),
                'created_at' => now()
            ]);

            if ($request->filled('models')) {
                foreach ($request->models as $model) {
                    $existModel = DB::table('car_models')
                        ->where('slug', Str::slug($model))->first();
                    if ($existModel && $existModel->id) {
                        DB::table('car_models')
                            ->where('id', $existModel->id)
                            ->update(['deleted_at' => null]);
                    } else {
                        DB::table('car_models')->insert([
                            'brand_id' => $brandId,
                            'name' => $model,
                            'slug' => Str::slug($model),
                            'created_at' => now()
                        ]);
                    }
                }
            }

            DB::commit();

            return $this->sendResponse(['data' => $brandId], 'Brand created', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    public function editBranch(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
            'name' => 'required|string',
            'models' => 'nullable|array',
            'models.*' => 'required|string'
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        DB::beginTransaction();

        try {
            $brandId = DB::table('car_brands')
                ->where('id', $request->id)
                ->update([
                    'name' => $request->name,
                    'is_active' => $request->is_active,
                    'slug' => Str::slug($request->name),
                    'updated_at' => now(),
                ]);

            if ($request->filled('models')) {
                DB::table('car_models')
                    ->where('brand_id', $request->id)
                    ->update(['deleted_at' => now()]);
                foreach ($request->models as $model) {
                    $existModel = DB::table('car_models')
                        ->where('slug', Str::slug($model))->first();
                    if ($existModel && $existModel->id) {
                        DB::table('car_models')
                            ->where('id', $existModel->id)
                            ->update(['deleted_at' => null]);
                    } else {
                        DB::table('car_models')->insert([
                            'brand_id' => $request->id,
                            'name' => $model,
                            'slug' => Str::slug($model),
                            'updated_at' => now(),
                        ]);
                    }
                }
            }

            DB::commit();

            return $this->sendResponse(['data' => $brandId], 'Brand created', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    public function disabledOrDeleteBranch(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {

            $brand = DB::table('car_brands')->where('id', $request->id)->first();

            if ($request->delete) {
                $brandId = DB::table('car_brands')
                    ->where('id', $request->id)
                    ->update(['deleted_at' => now()]);
            } else {

                $brandId = DB::table('car_brands')
                    ->where('id', $request->id)
                    ->update([
                        'is_active' => $brand->is_active === 0 ? 1 : 0,
                    ]);
            }

            return $this->sendResponse(['data' => $brandId], 'Brand created', 201);
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }
}
