<?php

namespace App\Http\Controllers;

use App\Models\User;
use Google_Client;
use Google_Service_Oauth2;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Facades\JWTAuth;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        // Google / Social signup
        if ($request->signUpWith) {
            if ($token = $this->registerWithGoogle($request)) {
                if ($token === 'exists') {
                    return $this->sendError('Email already exists', 'Processing Error', 422);
                }

                return $this->sendResponse(['token' => $token], 'User registered successfully', 201);
            }

            return $this->sendError('Registration failed! Please try again later.', 'Processing Error', 422);
        }

        // Email/password signup
        $input = $request->only(
            'sname',
            'oname',
            'role',
            'email',
            'password',
            'cpassword',
            'phone',
            'settings',
        );

        $validator = Validator::make($input, [
            'sname' => 'required|string|regex:/^[a-zA-Z0-9]+$/|max:255',
            'oname' => 'required|string|regex:/^[a-zA-Z0-9]+$/|max:255',
            'role' => 'nullable|in:learner,instructor,admin,proctor,enterprise,api_client',
            'settings' => 'nullable|array',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:50',
            'password' => 'required|string|min:8',
            'cpassword' => 'required|string|same:password',
        ]);

        if ($validator->fails()) {
            return $this->sendError($validator->errors()->first(), 'Validation Error', 422);
        }

        // Hash password and prepare data for Eloquent
        $userData = [
            'sname' => $input['sname'], // combine surname and other name
            'oname' => $input['oname'],
            'role' => $input['role'] ?? 'learner',
            'email' => $input['email'],
            'phone' => $input['phone'] ?? null,
            'password' => bcrypt($input['password']),
            'settings' => isset($input['settings']) ? json_encode($input['settings']) : null,
        ];

        $user = User::create($userData);

        if (! $user) {
            return $this->sendError('Registration failed! Please try again later.', 'Processing Error', 422);
        }

        // Generate JWT token
        $token = JWTAuth::attempt(['email' => $input['email'], 'password' => $input['password']]);

        return $this->sendResponse(['token' => $token], 'User registered successfully', 201);
    }

    public function addUser(Request $request)
    {
        $input = $request->only('sname', 'oname', 'username', 'email');
        $input['role'] = 3;
        $validator = Validator::make($input, [
            'sname' => 'required|string|max:255', // Ensure surname is required and a string
            'oname' => 'required|string|max:255', // Ensure other name is required and a string
            'username' => 'required|string|max:255|unique:users', // Remove invalid `username` rule
            'email' => $request->has('email') && $request->email != ''
                ? 'required|email|unique:users'
                : 'nullable|email', // Ensure nullable email is valid
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        DB::beginTransaction();

        $user_id = ! $this->checkJWTAuth() ? 0 : $this->JWTAuth()->id;
        $input['password'] = bcrypt('12345678'); // use bcrypt to hash the passwords

        if ($student = User::create($input)) {
            DB::commit();
            $success['data'] = $student;

            return $this->sendResponse($success, 'successfully', 201);
        } else {
            DB::rollBack();
            $error['data'] = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        }
    }

    public function addBulkStudents(Request $request)
    {
        $input = $request->only('students');
        $validator = Validator::make($input, [
            'students' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        $user_id = ! $this->checkJWTAuth() ? 0 : $this->JWTAuth()->id;
        $bulkStudents = json_decode($request->students);
        $students = [];
        $student_error = [];
        if (is_array($bulkStudents)) {
            foreach ($bulkStudents as $val) {
                $std_data = [
                    'sname' => $val->sname,
                    'oname' => $val->oname,
                    'username' => $val->username,
                    'email' => $val->email,
                    'role' => 'learner',
                ];

                $validator = Validator::make($std_data, [
                    'sname' => 'required|string|max:255', // Ensure surname is required and a string
                    'oname' => 'required|string|max:255', // Ensure other name is required and a string
                    'username' => 'required|string|max:255|unique:users', // Remove invalid `username` rule
                    'email' => isset($std_data['email']) && $std_data['email'] != ''
                        ? 'required|email|unique:users'
                        : 'nullable|email', // Ensure nullable email is valid
                ]);

                if (! $validator->errors()->first()) {
                    DB::beginTransaction();
                    $std_data['password'] = bcrypt('12345678'); // use bcrypt to hash the passwords

                    if ($student = User::create($std_data)) {
                        $follow_id = DB::table('user_followers')->insertGetId([
                            'following_id' => $user_id,
                            'follower_id' => $student->id,
                            'created_at' => \Carbon\Carbon::now(),
                        ]);

                        $countData = DB::table('total_user_followers')
                            ->where('user_id', $request->id)
                            ->update([
                                'follower' => DB::raw('follower + 1'),
                                'updated_at' => \Carbon\Carbon::now(),
                            ]);

                        $countData = DB::table('total_user_followers')
                            ->where('user_id', $this->JWTAuth()->id)
                            ->update([
                                'following' => DB::raw('following + 1'),
                                'updated_at' => \Carbon\Carbon::now(),
                            ]);
                    }
                    if ($student && $follow_id && $countData) {
                        DB::commit();
                        $students[] = $student;
                    } else {
                        DB::rollBack();
                    }
                } else {
                    $student_error[$val->username ?? $val->email] = $validator->errors()->first();
                }
            }
        }
        if ($students) {
            $success['data'] = ['success' => true, 'error' => $student_error];

            return $this->sendResponse($success, 'successfully', 201);
        } else {
            $error['data'] = $student_error;

            return $this->sendError($error, 'Validation Error', 422);
        }
    }

    public function login(Request $request)
    {
        if ($request->loginWith) {
            if ($user = $this->loginWithGoogle($request)) {
                $success = [
                    'token' => $user['token'],
                ];

                return $this->sendResponse($success, 'successful login', 200);
            } else {
                return $this->sendError([], 'invalid login credentials', 422);
            }
        } else {
            $input = $request->only('email', 'password');

            $validator = Validator::make($input, [
                'email' => 'required',
                'password' => 'required',
            ]);

            if ($validator->fails()) {
                return $this->sendError(
                    $validator->errors()->first(),
                    'Validation Error',
                    422
                );
            }

            try {
                // this authenticates the user details with the database and generates a token
                if (! ($token = JWTAuth::attempt($input))) {
                    $user = $this->loginWithUsername($request);
                    if ($user) {
                        $token = $user['token'];
                    } else {
                        return $this->sendError(
                            [],
                            'invalid login credentials',
                            400
                        );
                    }
                }
            } catch (JWTException $e) {
                return $this->sendError([], $e->getMessage(), 500);
            }

            $success = [
                'token' => $token,
            ];

            return $this->sendResponse($success, 'successful login', 200);
        }
    }

    public function loginWithUsername(Request $request)
    {
        $input = $request->only('email', 'password');

        if ($user = User::where('username', $input['email'])->first()) {
            if (Hash::check($input['password'], $user->password)) {
                // Generate a token for the user
                $token = JWTAuth::fromUser($user);
                if ($token) {
                    return ['token' => $token];
                }

                return false;
            }

            return false;
        }
    }

    public function getUser()
    {
        try {
            $user = JWTAuth::parseToken()->authenticate();
            if (! $user) {
                return $this->sendError([], 'user not found', 403);
            }
        } catch (JWTException $e) {
            return $this->sendError([], $e->getMessage(), 500);
        }

        if ($user->role === 'instructor') {
            $row = DB::table('organizations')->where('author_id', $user->id)->first();
            $user->firstOrg = $row;
        }

        return $this->sendResponse(
            ['user' => $user],
            'user data retrieved',
            200
        );
    }

    public function loginWithGoogle(Request $request)
    {
        $input = $request->only('googleAccessToken');
        $validator = Validator::make($input, [
            'googleAccessToken' => 'required',
        ]);
        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        if (
            $payload = $this->verifyGoogleCredentials(
                $request->googleAccessToken
            )
        ) {
            if ($user = User::where('email', $payload['email'])->first()) {
                // eloquent creation of data
                $token = JWTAuth::fromUser($user);
                if ($token) {
                    return ['token' => $token];
                }
            } else {
                if ($token = $this->registerWithGoogle($request)) {
                    return ['token' => $token];
                }

                return false;
            }
        } else {
            return false;
        }
    }

    public function registerWithGoogle(Request $request)
    {
        $input = $request->only('googleAccessToken');
        $validator = Validator::make($input, [
            'googleAccessToken' => 'required',
        ]);
        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        if (
            $payload = $this->verifyGoogleCredentials(
                $request->googleAccessToken
            )
        ) {
            DB::beginTransaction();
            $inputData = [
                'email' => $payload['email'],
                'sname' => $payload['name'],
                'oname' => $payload['family_name'],
            ];
            if (User::where('email', $payload['email'])->exists()) {
                DB::rollBack();

                return 'exists';
            }

            if ($user = User::create($inputData)) {
                // eloquent creation of data
                $token = JWTAuth::fromUser($user);
            }
            if ($token) {
                DB::commit();

                return $token;
            } else {
                DB::rollBack();

                return false;
            }
        } else {
            DB::rollBack();

            return false;
        }
    }

    public function verifyGoogleCredentials($token)
    {
        $googleClient = new Google_Client;
        $googleClient->setClientId(
            '361111829353-q6e4irgruujq3hk6sot19ur7sagauotl.apps.googleusercontent.com'
        );
        $googleClient->setClientSecret('Yej6rLXVywkMmvN6MSL_Sxxe');
        // Specify the redirect URI and other configurations as needed
        // $googleClient->setRedirectUri(...);
        // $googleClient->addScope(...);
        $payload = $googleClient->verifyIdToken($token);
        if ($payload) {
            // Example of fetching additional user information
            // $googleService = new Google_Service_Oauth2($googleClient);
            // $userInfo = $googleService->userinfo->get();
            return $payload;
            // return response()->json(['success' => $payload], 200);
        } else {
            return false;
            // return response()->json(['error' => 'Invalid access token'], 401);
        }
    }

    public function updateUser(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'sname' => 'nullable|min:3|max:240',
            'oname' => 'nullable|min:3|max:240',
            'country' => 'nullable|min:1|max:240',
            'state' => 'nullable|min:3|max:240',
            'town' => 'nullable|min:3|max:240',
            'gender' => 'nullable|min:1|max:240',
            'profile_img' => $request->file('profile_img') && $request->profile_img != ''
                ? 'required|image|mimes:jpg,png,jpeg'
                : 'nullable',
            'username' => [
                'nullable',
                'alpha_dash',
                'max:100',
                function ($attribute, $value, $fail) {
                    if (
                        auth()->user()->username !== $value &&
                        DB::table('users')->where('username', $value)->exists()
                    ) {
                        $fail('Username already exists');
                    }
                },
            ],
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        $user = User::find($this->JWTAuth()->id);

        $updateData = DB::table('users')
            ->where('id', $this->JWTAuth()->id)
            ->update([
                'sname' => $request->sname ? $request->sname : $user->sname,
                'sname' => $request->oname ? $request->sname : $user->oname,
                'username' => $request->username ? $request->username : $user->username,
                'country' => $request->country ? $request->country : $user->country,
                'state' => $request->state ? $request->state : $user->state,
                'town' => $request->town ? $request->town : $user->town,
                'gender' => $request->gender ? $request->gender : $user->gender,
                'updated_at' => \Carbon\Carbon::now(),
            ]);

        if ($updateData && $request->has('profile_img')) {
            if ($request->file('profile_img') && $request->profile_img != '') {
                $media_file = $request
                    ->file('profile_img')
                    ->store('images/profile_imgs', 'public');
            }

            DB::table('users')->where('id', $this->JWTAuth()->id)->update([
                'profile_img' => $media_file,
            ]);
        }

        if ($updateData) {
            $success['data'] = User::find($this->JWTAuth()->id);

            return $this->sendResponse(
                $success,
                'Profile updated successfully',
                201
            );
        }

        $error = 'something went wrong';

        return $this->sendError($error, 'Processing Error', 422);
    }

    public function changeUserPassword(Request $request)
    {
        $input = $request->only('password', 'cpassword', 'old_password');
        $user = User::find($this->JWTAuth()->id);

        $validator = Validator::make($input, [
            'password' => 'required|min:3|max:240',
            'cpassword' => 'required|same:password|max:240',
            'old_password' => [
                'required',
                function ($attribute, $value, $fail) use ($user) {
                    if (! Hash::check($value, $user->password)) {
                        return $fail(__('The current password is incorrect.'));
                    }
                },
            ],
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        $input['password'] = bcrypt($input['password']);
        $updatePassword = $user->update($input);
        if ($updatePassword) {
            $success['data'] = $updatePassword;

            return $this->sendResponse(
                $success,
                'Profile updated successfully',
                201
            );
        }
        $error = 'something went wrong';

        return $this->sendError($error, 'Processing Error', 422);
    }

    public function forgotPassword(Request $request)
    {
        $input = $request->only('email');
        $validator = Validator::make($input, [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        $status = Password::sendResetLink($request->only('email'));

        if ($status === Password::RESET_LINK_SENT) {
            $success['data'] = $status;

            return $this->sendResponse(
                $success,
                'Password reset link sent!',
                201
            );
        }
        $error = 'Unable to send reset link';

        return $this->sendError($error, 'Processing Error', 422);
    }

    public function resetPassword(Request $request)
    {
        $input = $request->only('email', 'token', 'password');
        $validator = Validator::make($input, [
            'email' => 'required|email',
            'token' => 'required',
            'password' => 'required|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        $status = Password::reset(
            $request->only(
                'email',
                'password',
                'password_confirmation',
                'token'
            ),
            function ($user, $password) {
                $user
                    ->forceFill([
                        'password' => Hash::make($password),
                        'remember_token' => Str::random(60),
                    ])
                    ->save();

                event(new PasswordReset($user));
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            $success['data'] = $status;

            return $this->sendResponse(
                $success,
                'Password has been reset!',
                201
            );
        }
        $error = 'Invalid token or email.';

        return $this->sendError($error, 'Processing Error', 422);
    }

    public function resetPasswordByAdmin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        $user = User::find($request->id);

        $input['password'] = bcrypt('12345678');

        $updatePassword = $user->update($input);
        if ($updatePassword) {
            $success['data'] = $updatePassword;

            return $this->sendResponse(
                $success,
                'Profile updated successfully',
                201
            );
        }
        $error = 'something went wrong';

        return $this->sendError($error, 'Processing Error', 422);

        if ($updatePassword) {
            DB::commit();
            $success['data'] = $updatePassword;

            return $this->sendResponse($success, 'successful', 201);
        } else {
            DB::rollBack();
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        }
    }

    public function logout(Request $req)
    {
        try {
            $token = JWTAuth::getToken();
            JWTAuth::invalidate($token);
            $success['data'] = 'Logged out';

            return $this->sendResponse($success, 'successful', 201);
        } catch (\Exception $e) {
            $error = 'something went wrong';

            return $this->sendError($error, 'Could not invalidate token', 422);
        }
    }

    public function me(Request $req)
    {
        $user = JWTAuth::parseToken()->authenticate();
        $success = [
            'user' => $user,
        ];

        return $this->sendResponse($success, 'successful login', 200);
    }
}
